// OBD-II Viewer.cpp : Defines the entry point for the console application.
//

#include "stdafx.h"

#include "PCAN-OBDII.h"


///////////////////////////////////////////////////
// Utility functions to output OBDII structures

#define PRINT_BUFFER	5120

// Outputs a TPOBDIIResponse structure
DWORD OBDIIPrintResponse(TPOBDIIResponse& Data, bool doReturn = true)
{
	/* Output Data to String */
	CHAR String[PRINT_BUFFER]="";
	DWORD pos = 0;
	pos = sprintf_s(String+pos, PRINT_BUFFER-pos, "    * SOURCE=%02X, ", Data.SOURCE);
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      ERRORNR=%02X, ", Data.ERRORNR);
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      SID=%02X, ", Data.SID);
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      PID=%02X, ", Data.PID);
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      FRAME=%02X, ", Data.FRAME);
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      RAW_DATA[%d]=[", Data.LEN);
	for (int r = 0; r < Data.LEN; ++r < Data.LEN && (pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, ", "))){
		pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "%02X", Data.DATA[r]);		
	}
	if (doReturn)
		pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "]\n");
	else
		pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "], ");


	printf("%s", String);
	return 0;
}

// Outputs a TPOBDIIParamData structure
DWORD OBDIIPrintDataPID(TPOBDIIParamData& Data)
{
	/* Output Data to String */
	CHAR String[PRINT_BUFFER]="";
	DWORD pos = 0;
	
	OBDIIPrintResponse(Data.RESPONSE, false);
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      BUFFER[%d]=[", Data.BLEN);
	for (int r = 0; r < Data.BLEN; ++r < Data.BLEN && (pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, ", "))){
		if (Data.BYTEMASK & (1 << r))
			pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "%02X", Data.BUFFER[r]);
		else
			pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "%02x", Data.BUFFER[r]);//, "xx");
	}
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "]\n");
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      BYTEMASK=%03X\n", Data.BYTEMASK);
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      DOUBLES[%d]=[", Data.DLEN);
	for (int r = 0; r < Data.DLEN; ++r < Data.DLEN && (pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, ", "))){
		pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "%13.6lf", Data.DOUBLES[r]);
	}
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "]\n");
	printf("%s", String);
	return 0;
}

// Outputs a TPOBDIIDTCData structure
DWORD OBDIIPrintDataDTC(TPOBDIIDTCData& Data)
{
	/* Output Data to String */
	CHAR String[PRINT_BUFFER]="";
	DWORD pos = 0;
	OBDIIPrintResponse(Data.RESPONSE, false);
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      DTC[%d]=[", Data.DLEN);
	for (int r = 0; r < Data.DLEN; ++r < Data.DLEN && (pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, ", "))){
		pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "%0.6s", Data.DTC[r]);
	}
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "]\n");
	printf("%s", String);
	return 0;
}

// Outputs a TPOBDIIMonitorData structure
DWORD OBDIIPrintDataMID(TPOBDIIMonitorData& Data)
{
	/* Output Data to String */
	CHAR String[PRINT_BUFFER]="";
	TPOBDIIUnitAndScaling us;
	OBDII_GetUnitAndScaling(Data.UNITANDSCALING, &us);

	DWORD pos = 0;
	OBDIIPrintResponse(Data.RESPONSE, false);
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      TID=%02X, ", Data.TID);
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      UnS=%02X (%s)", Data.UNITANDSCALING, us.UNIT);
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      TV=%13.6lf, ", Data.TESTVALUE);
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      MIN=%13.6lf, ", Data.MINLIMIT);
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      MAX=%13.6lf\n", Data.MAXLIMIT);
	printf("%s", String);
	return 0;
}

// Outputs a TPOBDIIInfoData structure
DWORD OBDIIPrintDataITp(TPOBDIIInfoData& Data)
{
	/* Output Data to String */
	CHAR String[PRINT_BUFFER]="";
	DWORD pos = 0;
	OBDIIPrintResponse(Data.RESPONSE, false);
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      INDEXNR=%02X, ", Data.INDEXNR);
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      DATATYPE=%02X, ", Data.DATATYPE);
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      COUNTER=%05u, ", Data.COUNTER);
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      CALDATA[%d]=[", 4);
	for (int r = 0; r < 4; ++r < 4 && (pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, ", "))){
		pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "%02X", Data.CALDATA[r]);
	}
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "], ");
	pos+= sprintf_s(String+pos, PRINT_BUFFER-pos, "      STRING=%s\n", Data.TEXT);
	printf("%s", String);
	return 0;
}



// Tests OBDII to GetValue of supported PIDs
void GetSupportedMask(TPOBDIICANHandle Channel)
{
	TPOBDIIStatus status;

	// Will Read Supported IDs for specific services
	BYTE Support[5][257] = {};
	// supported PIDs for Service 01: Current Data
	status = OBDII_GetValue(Channel, POBDII_PARAM_SUPPORTMASK_PIDS, Support[0], 256);
	// supported OBDMIDs for Service 06: Test Results
	status = OBDII_GetValue(Channel, POBDII_PARAM_SUPPORTMASK_OBDMIDS, Support[1], 256);
	// supported TIDs for Service 08: Control Operation
	status = OBDII_GetValue(Channel, POBDII_PARAM_SUPPORTMASK_TIDS, Support[2], 256);
	// supported InfoTypes for Service 09: Vehicle Information
	status = OBDII_GetValue(Channel, POBDII_PARAM_SUPPORTMASK_INFOTYPES, Support[3], 256);
	// supported FFIDs for Service 02: Freeze Frame Data 
	Support[4][0] = 0;
	status = OBDII_GetValue(Channel, POBDII_PARAM_SUPPORTMASK_FFPIDS, Support[4], 257);

	// Output support for PIDs, TIDs, OBDMIDs and InfoTypes
	printf("Number of Supported PIDs, TIDs, OBDMIDs and InfoTypes:\n");
	printf("(values are in binary format where a bit corresponds to a specific ECU)\n\n");
	printf("     | PID | FFID | MID | TID | IT |\n");
	for (int i = 0; i < 256; i++){
		// checks if there are subsequent ranges of supported IDs (ranges are 0x20 long)
		if ((i % 0x20) == 0){
			if (Support[0][i] || Support[1][i] || Support[2][i] || Support[3][i] || Support[4][i+1]) {
				printf("  ---+------+-----+-----+-----+-----+\n");
			}else{
				// not any more supported IDs, ref. SAE-J1979 APPENDIX A:
				// the ECU shall not respond to unsupported PID/OBDMID/TID/InfoType ranges 
				// unless subsequent ranges have a supported PID/OBDMID/TID/InfoType.					
				break;
			}
		}
		if (Support[0][i] || Support[1][i] || Support[2][i] || Support[3][i] || Support[4][i+1]) {
			printf("  %02x |  %02x  |  %02x |  %02x |  %02x |  %02x |\n", i,
				Support[0][i],
				Support[4][i+1],
				Support[1][i],
				Support[2][i],
				Support[3][i]);
		}
	}
	printf("  ---+------+-----+-----+-----+-----+\n");	

	printf("\nPress <Enter> to continue...\n");
	getchar();
}



// Gets and displays the supported IDs for an OBDII Service
TPOBDIIStatus getSupportMask(TPOBDIICANHandle Channel, BYTE param, BYTE * buffer, WORD bufferLength)
{	
	TPOBDIIStatus status;
	int i, offset = 0;
	// Get supported IDs
	status = OBDII_GetValue(Channel, param, buffer, bufferLength);
	if (status != POBDII_ERROR_OK) 
		return status;

	if (param == POBDII_PARAM_SUPPORTMASK_FFPIDS) 
		offset = 1;
	// Output supported IDs for requested service
	printf(" Number of Supported IDs:\n");	
	for (i = 0 ; i < bufferLength - offset ; i++)
	{
		// checks if there are subsequent ranges of supported IDs (ranges are 0x20 long)
		if ((i % 0x20) == 0){
			if (buffer[i + offset]) 
				printf(" ---+-----+\n");
			else 
			{
				// not any more supported IDs
				i = 256 * ((i/256) + 1) - 1;
				continue;
			}
		}
		if (buffer[i+offset]) 
			printf(" %02x |  %02x |\n", i%256, buffer[i + offset]);
	}
	if (i == 0)
		printf("    \\-> NONE.\n");
	else
		printf(" ---+-----+\n");

	return status;	
}

// Display supported PIDs and values for OBD mode 01
int displayService01(TPOBDIICANHandle Channel)
{
	TPOBDIIStatus status;
	WORD bufferLength = 256;
	BYTE support[256] = {};
	WORD idx;
	BYTE dataLength = 8;
	TPOBDIIParamData dataPIDArray[8];
	int nbError = 0;
	
	printf("\n\nSERVICE: %02X, RequestCurrentData\n", 0x01);
	status = getSupportMask(Channel, POBDII_PARAM_SUPPORTMASK_PIDS, support, bufferLength);
	for (idx = 0; idx < bufferLength ; idx++) {
		// check if PID is supported
		if (support[idx] == 0)
			continue;
		status = OBDII_RequestCurrentData(Channel, (BYTE) idx, dataPIDArray, dataLength);
		if (status == POBDII_ERROR_OK) 
		{
			if (dataPIDArray[0].RESPONSE.ERRORNR != POBDII_R_NOT_USED)
				printf("  PID: %02X\n", idx);
			for (int i = 0 ; i < dataLength ; i++)
			{
				if (dataPIDArray[i].RESPONSE.ERRORNR == POBDII_R_NOT_USED) 
					break;
				OBDIIPrintDataPID(dataPIDArray[i]);
			}
		}
		else if (status == POBDII_ERROR_NO_MESSAGE) 
		{
			printf("  No responses..\n");
		}
		else
		{
			printf("  PID: %02X: ERROR (%d) !!!\n", idx, status);
			nbError++;
		}
	}

	return nbError;
}

// Display supported IDs and values for OBD mode 02
int displayService02(TPOBDIICANHandle Channel)
{
	TPOBDIIStatus status;
	WORD bufferLength = 257;
	BYTE supportWithFrame[257] = {};
	BYTE * support;
	WORD idx;
	BYTE dataLength = 8;
	TPOBDIIParamData dataPIDArray[8];
	int nbError = 0;
	
	printf("\n\nSERVICE: %02X, RequestFreezeFrameData\n", 0x02);
	status = getSupportMask(Channel, POBDII_PARAM_SUPPORTMASK_FFPIDS, supportWithFrame, bufferLength);
	support = &supportWithFrame[1];
	for (idx = 0; idx < bufferLength ; idx++) 
	{
		// check if PID is supported
		if (support[idx] == 0)
			continue;
		status = OBDII_RequestFreezeFrameData(Channel, (BYTE) idx, 0, dataPIDArray, dataLength);
		if (status == POBDII_ERROR_OK) {
			if (dataPIDArray[0].RESPONSE.ERRORNR != POBDII_R_NOT_USED)
				printf("  FFID: %02X\n", idx);
			for (int i = 0 ; i < dataLength ; i++)
			{
				if (dataPIDArray[i].RESPONSE.ERRORNR == POBDII_R_NOT_USED) 
					break;
				OBDIIPrintDataPID(dataPIDArray[i]);
			}
		}
		else if (status == POBDII_ERROR_NO_MESSAGE) 
		{
			printf("  No responses..\n");
		}
		else
		{
			printf("  FFID: %02X: ERROR (%d) !!!\n", idx, status);
			nbError++;
		}	
	}

	return nbError;
}
// Display supported IDs and values for OBD mode 03
int displayService03(TPOBDIICANHandle Channel)
{
	TPOBDIIStatus status;
	WORD bufferLength = 256;
	BYTE support[256] = {};
	BYTE dataLength = 8;
	TPOBDIIDTCData dataDTCArray[8];
	int nbError = 0;
	
	printf("\n\nSERVICE: %02X, RequestStoredTroubleCodes\n", 0x03);
	status = OBDII_RequestStoredTroubleCodes(Channel, dataDTCArray, dataLength);
	if (status == POBDII_ERROR_OK) 
	{
		for (int i = 0 ; i < dataLength ; i++)
		{
			if (dataDTCArray[i].RESPONSE.ERRORNR == POBDII_R_NOT_USED) 
				break;
			OBDIIPrintDataDTC(dataDTCArray[i]);
		}
	}
	else if (status == POBDII_ERROR_NO_MESSAGE) 
	{
		printf("  No responses..\n");
	}
	else
	{
		printf("  ERROR (%d) !!!\n", status);
		nbError++;
	}

	return nbError;
}
// Display supported IDs and values for OBD mode 04
int displayService04(TPOBDIICANHandle Channel)
{
	TPOBDIIStatus status;
	WORD bufferLength = 256;
	BYTE support[256] = {};
	BYTE dataLength = 8;
	TPOBDIIResponse responseArray[8];
	int nbError = 0;
	
	printf("\n\nSERVICE: %02X, ClearTroubleCodes\n", 0x04);
	status = OBDII_ClearTroubleCodes(Channel, responseArray, dataLength);
	if (status == POBDII_ERROR_OK) 
	{
		for (int i = 0 ; i < dataLength ; i++)
		{
			if (responseArray[i].ERRORNR == POBDII_R_NOT_USED) 
				break;
			OBDIIPrintResponse(responseArray[i]);
		}
	}
	else if (status == POBDII_ERROR_NO_MESSAGE) 
	{
		printf("  No responses..\n");
	}
	else
	{
		printf("  ERROR (%d) !!!\n", status);
		nbError++;
	}

	return nbError;
}
// Display supported IDs and values for OBD mode 06
int displayService06(TPOBDIICANHandle Channel)
{
	TPOBDIIStatus status;
	WORD bufferLength = 256;
	BYTE support[256] = {};
	WORD idx;
	BYTE dataLength = 20;
	TPOBDIIMonitorData dataMIDArray[20];
	int nbError = 0;
	
	
	printf("\n\nSERVICE: %02X, RequestTestResults\n", 0x06);
	status = getSupportMask(Channel, POBDII_PARAM_SUPPORTMASK_OBDMIDS, support, bufferLength);
	for (idx = 0; idx < bufferLength ; idx++) 
	{
		// check if PID is supported
		if (support[idx] == 0)
			continue;
		status = OBDII_RequestTestResults(Channel, (BYTE) idx, dataMIDArray, dataLength);
		if (status == POBDII_ERROR_OK) 
		{
			if (dataMIDArray[0].RESPONSE.ERRORNR != POBDII_R_NOT_USED)
				printf("  MID: %02X\n", idx);
			for (int i = 0 ; i < dataLength ; i++)
			{
				if (dataMIDArray[i].RESPONSE.ERRORNR == POBDII_R_NOT_USED) 
					break;
				OBDIIPrintDataMID(dataMIDArray[i]);
			}
		}
		else if (status == POBDII_ERROR_NO_MESSAGE) 
		{
			printf("  No responses..\n");
		}
		else
		{
			printf("  MID: %02X: ERROR (%d) !!!\n", idx, status);
			nbError++;
		}
	}

	return nbError;
}
// Display supported IDs and values for OBD mode 07
int displayService07(TPOBDIICANHandle Channel)
{
	TPOBDIIStatus status;
	WORD bufferLength = 256;
	BYTE support[256] = {};
	BYTE dataLength = 8;
	TPOBDIIDTCData dataDTCArray[8];
	int nbError = 0;
	
	printf("\n\nSERVICE: %02X, RequestPendingTroubleCodes\n", 0x07);
	status = OBDII_RequestPendingTroubleCodes(Channel, dataDTCArray, dataLength);
	if (status == POBDII_ERROR_OK) 
	{
		for (int i = 0 ; i < dataLength ; i++)
		{
			if (dataDTCArray[i].RESPONSE.ERRORNR == POBDII_R_NOT_USED) 
				break;
			OBDIIPrintDataDTC(dataDTCArray[i]);
		}
	}
	else if (status == POBDII_ERROR_NO_MESSAGE) 
	{
		printf("  No responses..\n");
	}
	else
	{
		printf("  ERROR (%d) !!!\n", status);
		nbError++;
	}

	return nbError;
}
// Display supported IDs and values for OBD mode 08
int displayService08(TPOBDIICANHandle Channel)
{
	TPOBDIIStatus status;
	WORD bufferLength = 256;
	BYTE support[256] = {};
	WORD idx;
	BYTE dataLength = 8;
	TPOBDIIResponse ResponseArray[8];
	int nbError = 0;	

	printf("\n\nSERVICE: %02X, RequestControlOperation\n", 0x08);
	status = getSupportMask(Channel, POBDII_PARAM_SUPPORTMASK_TIDS, support, bufferLength);
	for (idx = 0; idx < bufferLength ; idx++) 
	{
		// check if PID is supported
		if (support[idx] == 0)
			continue;
		status = OBDII_RequestControlOperation(Channel, (BYTE) idx, ResponseArray, dataLength);
		if (status == POBDII_ERROR_OK) 
		{
			if (ResponseArray[0].ERRORNR != POBDII_R_NOT_USED)
				printf("TID: %02X\n", idx);
			for (int i = 0 ; i < dataLength ; i++)
			{
				if (ResponseArray[i].ERRORNR == POBDII_R_NOT_USED) 
					break;
				OBDIIPrintResponse(ResponseArray[i]);
			}
		}
		else if (status == POBDII_ERROR_NO_MESSAGE) 
		{
			printf("  No responses..\n");
		}
		else
		{
			printf("  TID: %02X: ERROR (%d) !!!\n", idx, status);
			nbError++;
		}
	}

	return nbError;
}
// Display supported IDs and values for OBD mode 09
int displayService09(TPOBDIICANHandle Channel)
{
	TPOBDIIStatus status;
	WORD bufferLength = 256;
	BYTE support[256] = {};
	WORD idx;
	BYTE dataLength = 20;
	TPOBDIIInfoData dataITArray[20];
	int nbError = 0;

	printf("\n\nSERVICE: %02X, RequestVehicleInformation\n", 0x09);
	status = getSupportMask(Channel, POBDII_PARAM_SUPPORTMASK_INFOTYPES, support, bufferLength);
	for (idx = 0; idx < bufferLength ; idx++) 
	{
		// check if IT is supported
		if (support[idx] == 0)
			continue;
		status = OBDII_RequestVehicleInformation(Channel, (BYTE) idx, dataITArray, dataLength);
		if (status == POBDII_ERROR_OK) 
		{
			if (dataITArray[0].RESPONSE.ERRORNR != POBDII_R_NOT_USED)
				printf("  PID: %02X\n", idx);
			for (int i = 0 ; i < dataLength ; i++)
			{
				if (dataITArray[i].RESPONSE.ERRORNR == POBDII_R_NOT_USED) 
					break;
				OBDIIPrintDataITp(dataITArray[i]);
			}
		}
		else if (status == POBDII_ERROR_NO_MESSAGE) 
		{
			printf("  No responses..\n");
		}
		else
		{
			printf("  IT: %02X: ERROR (%d) !!!\n", idx, status);
			nbError++;
		}
	}

	return nbError;
}
// Display supported IDs and values for OBD mode 0A
int displayService0A(TPOBDIICANHandle Channel)
{
	TPOBDIIStatus status;
	WORD bufferLength = 256;
	BYTE support[256] = {};
	BYTE dataLength = 8;
	TPOBDIIDTCData dataDTCArray[8];
	int nbError = 0;	

	printf("\n\nSERVICE: %02X, RequestPermanentTroubleCodes\n", 0x0A);
	status = OBDII_RequestPermanentTroubleCodes(Channel, dataDTCArray, dataLength);
	if (status == POBDII_ERROR_OK) 
	{
		for (int i = 0 ; i < dataLength ; i++)
		{
			if (dataDTCArray[i].RESPONSE.ERRORNR == POBDII_R_NOT_USED) 
				break;
			OBDIIPrintDataDTC(dataDTCArray[i]);
		}
	}
	else if (status == POBDII_ERROR_NO_MESSAGE) 
	{
		printf("  No responses..\n");
	}
	else
	{
		printf("  ERROR (%d) !!!\n", status);
		nbError++;
	}

	return nbError;
}

// Main Entry Point
int _tmain(int argc, _TCHAR* argv[])
{
	TPOBDIICANHandle Channel;
	TPOBDIIStatus status;
	int nbErr = 0;
	char c = 0;
	
	// Set the PCAN-Channel to use (PCAN-USB Channel 1)
	Channel = POBDII_USBBUS1;
	// Define logging (of creation and destruction, and for newly created connections)
	BYTE debug = POBDII_LOGGING_TO_STDOUT;
	//OBDII_SetValue(POBDII_NONEBUS, POBDII_PARAM_LOGGING, &debug, 1);	
	printf("Channel: 0x%02x, CAN BUS Status: 0x%02x\n\n", Channel, OBDII_GetStatus(Channel));
	
	status = OBDII_Initialize(Channel);	
	if (status == POBDII_ERROR_OK) 
	{	
		BYTE baudrate = 0;
		BYTE canId = 0;
		
		OBDII_GetValue(Channel, POBDII_PARAM_BAUDRATE, &baudrate, 1);
		OBDII_GetValue(Channel, POBDII_PARAM_CAN_ID, &canId, 1);
		
		printf("*** OBD connection successfull [baudrate = %s, %d bit CAN identifier].\n\n", 
			baudrate == POBDII_BAUDRATE_250K ? "250 kbit/s" : "500 kbit/s",
			canId);

		// Request Vehicule Identification Number
		TPOBDIIInfoData dataIT;
		status = OBDII_RequestVehicleInformation(Channel, 0x02, &dataIT, 1);
		if (status == POBDII_ERROR_OK)
			printf("Vehicule Identification Number: '%s'\n\n", dataIT.TEXT);
		// Show supported IDS
		GetSupportedMask(Channel);

		// make requests to every supported services
		nbErr += displayService01(Channel);
		nbErr += displayService02(Channel);
		nbErr += displayService03(Channel);

		
		printf("\n\nDo you wish to Clear Diagnostic Trouble Codes ? (y/N) ");
		scanf_s("%c", &c, 1);
		if (c == 'y' || c == 'Y')
			nbErr += displayService04(Channel);
		while((c = getchar()) != '\n' && c != EOF)
			;/* discard */
		
		nbErr += displayService06(Channel);
		nbErr += displayService07(Channel);
		nbErr += displayService08(Channel);
		nbErr += displayService09(Channel);
		nbErr += displayService0A(Channel);
	}
	else
	{
		printf("Error (%d) !!!", status);
		nbErr++;
	}
	OBDII_Uninitialize(Channel);

	if (nbErr > 0) {
		printf("\n\nERROR : a total of %d errors occured.\n\n", nbErr);
	} 
	else {		
		printf("\n\nALL Transmissions succeeded !\n\n");
	}

	printf("\nPress <Enter> to quit...\n");
	getchar();
	return 0;
}